/*
 * Copyright (C) 2005 Serge Vakulenko, <vak@cronyx.ru>
 *
 * This file is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You can redistribute this file and/or modify it under the terms of the GNU
 * General Public License (GPL) as published by the Free Software Foundation;
 * either version 2 of the License, or (at your discretion) any later version.
 * See the accompanying file "COPYING" for more details.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include "audio.h"
#include "wav.h"

const char version[] = "1.0";
const char copyright[] = "Copyright (C) 2005 Serge Vakulenko";

char *progname;
int verbose;
int debug;
int wpm = 10;
int tone = 600;
int space_seen;
int dot_msec;
int dash_msec;
int pause_msec;

void (*play_sound) (int msec, int tone);

void type (char *str)
{
	if (*str++ == '.')
		play_sound (dot_msec, tone);
	else
		play_sound (dash_msec, tone);

	while (*str) {
		play_sound (dot_msec, 0);

		if (*str++ == '.')
			play_sound (dot_msec, tone);
		else
			play_sound (dash_msec, tone);
	}
}

void type_char (char c)
{
	if (isspace (c) || ! isprint (c))
		c = ' ';
	else if (isalpha (c) && islower (c))
		c = toupper (c);

	switch (c) {
	default:
		return;
	case ' ' :
		if (! space_seen)
			play_sound (dot_msec + pause_msec, 0);
		space_seen = 1;
		return;

	case 'A': type (".-"); break;
	case 'B': type ("-..."); break;
	case 'C': type ("-.-."); break;
	case 'D': type ("-.."); break;
	case 'E': type ("."); break;
	case 'F': type ("..-."); break;
	case 'G': type ("--."); break;
	case 'H': type ("...."); break;
	case 'I': type (".."); break;
	case 'J': type (".---"); break;
	case 'K': type ("-.-"); break;
	case 'L': type (".-.."); break;
	case 'M': type ("--"); break;
	case 'N': type ("-."); break;
	case 'O': type ("---"); break;
	case 'P': type (".--."); break;
	case 'Q': type ("--.-"); break;
	case 'R': type (".-."); break;
	case 'S': type ("..."); break;
	case 'T': type ("-"); break;
	case 'U': type ("..-"); break;
	case 'V': type ("...-"); break;
	case 'W': type (".--"); break;
	case 'X': type ("-..-"); break;
	case 'Y': type ("-.--"); break;
	case 'Z': type ("--.."); break;
	case '0': type ("-----"); break;
	case '1': type (".----"); break;
	case '2': type ("..---"); break;
	case '3': type ("...--"); break;
	case '4': type ("....-"); break;
	case '5': type ("....."); break;
	case '6': type ("-...."); break;
	case '7': type ("--..."); break;
	case '8': type ("---.."); break;
	case '9': type ("----."); break;
	case '.': type (".-.-.-"); break;
	case ',': type ("--..--"); break;
	case '?': type ("..--.."); break;
	case '/': type ("-..-."); break;
	case '=': type ("-...-"); break;
	}
	space_seen = 0;
}

void type_string (char *str)
{
	while (*str) {
		type_char (*str++);
		play_sound (pause_msec, 0);
	}

}

void type_file (char *filename)
{
	FILE *fd;
	int c;

	fd = fopen (filename, "r");
	if (! fd) {
		perror (filename);
		exit (-1);
	}
	/* Silence 1 sec at start. */
	play_sound (1000, 0);
	while ((c = getc (fd)) >= 0) {
		type_char (c);
		play_sound (pause_msec, 0);
	}
	/* Silence 1 sec at end. */
	play_sound (1000, 0);
	fclose (fd);
}

void usage ()
{
	fprintf (stderr, "Morse Echo, Version %s, %s\n", version, copyright);
	fprintf (stderr, "Usage:\n");
	fprintf (stderr, "\t%s [-vd] [-w wpm] [-e ewpm] [-t tone] [-f infile] [-o outfile] word...\n",
		progname);
	fprintf (stderr, "Options:\n");
	fprintf (stderr, "\t-v\tverbose mode\n");
	fprintf (stderr, "\t-d\tdebug\n");
	fprintf (stderr, "\t-w #\trate, default %d words per minute\n", wpm);
	fprintf (stderr, "\t-e #\teffective rate (less than wpm)\n");
	fprintf (stderr, "\t-t #\ttone frequency, default %d Hz\n", tone);
	fprintf (stderr, "\t-f #\tplay file\n");
	fprintf (stderr, "\t-o #\toutput to WAV file\n");
	exit (-1);
}

int main (int argc, char **argv)
{
	int i, effective_wpm = 0;
	char *infile = 0, *outfile = 0;;

	progname = *argv;
	for (;;) {
		switch (getopt (argc, argv, "vdw:e:t:f:o:")) {
		case EOF:
			break;
		case 'v':
			++verbose;
			continue;
		case 'd':
			++debug;
			continue;
		case 'w':
			wpm = strtol (optarg, 0, 0);
			continue;
		case 'e':
			effective_wpm = strtol (optarg, 0, 0);
			continue;
		case 't':
			tone = strtol (optarg, 0, 0);
			continue;
		case 'f':
			infile = optarg;
			continue;
		case 'o':
			outfile = optarg;
			continue;
		default:
			usage ();
		}
		break;
	}
	argc -= optind;
	argv += optind;

	if ((argc < 1 && ! infile) || wpm == 0 || effective_wpm > wpm)
		usage ();

	/* Compute timings. */
	dot_msec = 1200 / wpm;
	dash_msec = 3600 / wpm;
	if (effective_wpm) {
		/* Inrease pauses to lower the effective wpm. */
		pause_msec = 10000 / effective_wpm - 6400 / wpm;
	} else
		pause_msec = 3600 / wpm;

	/* Open audio device. */
	if (outfile) {
		wav_init (outfile);
		play_sound = wav_sound;
	} else {
		audio_init ();
		play_sound = audio_sound;
	}

	/* Play file. */
	if (infile)
		type_file (infile);

	/* Play args. */
	for (i=0; i<argc; ++i) {
		space_seen = 1;
		type_string (argv[i]);
		type_char (' ');
	}

	/* Close audio device. */
	if (outfile)
		wav_close ();
	else
		audio_flush ();
	return (0);
}
