/*
 * Copyright (C) 2005 Serge Vakulenko, <vak@cronyx.ru>
 *
 * This file is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You can redistribute this file and/or modify it under the terms of the GNU
 * General Public License (GPL) as published by the Free Software Foundation;
 * either version 2 of the License, or (at your discretion) any later version.
 * See the accompanying file "COPYING" for more details.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "wav.h"

#if defined (__ppc__)
#define LITTLE_ENDIAN_32(x)	((((x) >> 24) & 0xff) | (((x) >> 8) & 0xff00) | \
				 (((x) & 0xff00) << 8) | (((x) & 0xff) << 24))
#define LITTLE_ENDIAN_16(x)	((((x) >> 8) & 0xff) | (((x) & 0xff) << 8))
#endif

#ifndef LITTLE_ENDIAN_32
#define LITTLE_ENDIAN_32(x)	(x)
#endif

#ifndef LITTLE_ENDIAN_16
#define LITTLE_ENDIAN_16(x)	(x)
#endif

struct wav_header_t {
	unsigned long riff;
	unsigned long file_length;
	unsigned long wave;
	unsigned long fmt;
	unsigned long fmt_length;
	short fmt_tag;
	short channels;
	unsigned long sample_rate;
	unsigned long bytes_per_second;
	short block_align;
	short bits;
	unsigned long data;
	unsigned long data_length;
};

/* init with default values */
static struct wav_header_t wavhdr = {
	LITTLE_ENDIAN_32 (0x46464952),		/* "RIFF" */
	0,
	LITTLE_ENDIAN_32 (0x45564157),		/* "WAVE" */
	LITTLE_ENDIAN_32 (0x20746d66),		/* "fmt " */
	LITTLE_ENDIAN_32 (16),
	LITTLE_ENDIAN_16 (1),			/* PCM format */
	LITTLE_ENDIAN_16 (2),
	LITTLE_ENDIAN_32 (44100),
	LITTLE_ENDIAN_32 (192000),
	LITTLE_ENDIAN_16 (4),
	LITTLE_ENDIAN_16 (16),
	LITTLE_ENDIAN_32 (0x61746164),		/* "data" */
	0,
};

static FILE *fp;
static int sample_rate;

void wav_init (char *filename)
{
	int channels, bits;

	sample_rate = 8000;
	channels = 1;
	bits = 16;

	wavhdr.sample_rate = LITTLE_ENDIAN_32 (sample_rate);
	wavhdr.channels = LITTLE_ENDIAN_16 (channels);
	wavhdr.bits = LITTLE_ENDIAN_16 (bits);
	wavhdr.bytes_per_second = channels * sample_rate * bits / 8;
	wavhdr.bytes_per_second = LITTLE_ENDIAN_32 (wavhdr.bytes_per_second);
	wavhdr.data_length = LITTLE_ENDIAN_32 (0x7ffff000);
	wavhdr.file_length = wavhdr.data_length + sizeof (wavhdr) - 8;

	if (strcmp (filename, "-") == 0)
		fp = stdout;
	else {
		fp = fopen (filename, "wb");
		if (! fp) {
			perror (filename);
			exit (-1);
		}
	}

	/* Reserve space for wave header */
	fwrite (&wavhdr, sizeof (wavhdr), 1, fp);
	wavhdr.data_length = 0;
}

/* close audio device */
void wav_close ()
{
	if (fseek (fp, 0, SEEK_SET) == 0) {
		/* Write wave header */
		wavhdr.file_length = wavhdr.data_length + sizeof (wavhdr) - 8;
		wavhdr.file_length = LITTLE_ENDIAN_32 (wavhdr.file_length);
		wavhdr.data_length = LITTLE_ENDIAN_32 (wavhdr.data_length);
		fwrite (&wavhdr, sizeof (wavhdr), 1, fp);
	}
	fclose (fp);
}

static int fade (int n, int nsamples)
{
	int nfade;

	/* Fade 2 msec. */
	nfade = 2 * sample_rate / 1000;
	if (n < nfade)
		return 0x7fffL * n / nfade;
	if (n > nsamples - nfade)
		return 0x7fffL * (nsamples - n) / nfade;
	return 0x7fff;
}

void wav_sound (int msec, int freq)
{
	int phase;
	int n, nsamples;
	signed short sample;

	nsamples = msec * sample_rate / 1000;

	phase = 0;
	for (n=0; n<nsamples; ++n) {
		if (freq) {
			phase += freq;
			if (phase > sample_rate)
				phase -= sample_rate;
			sample = fade (n, nsamples) *
				sin (2*M_PI * phase/sample_rate);
			sample = LITTLE_ENDIAN_16 (sample);
		} else
			sample = 0;

		fwrite (&sample, sizeof (sample), 1, fp);
	}

	wavhdr.data_length += nsamples * sizeof (sample);
}
